<?php
// calcular_ponto.php
// Requisitos: conexao.php no mesmo diretório, tabelas: professores, escolas, ponto
include 'conexao.php';
date_default_timezone_set('America/Sao_Paulo'); // conforme sua timezone

// ---------------------- helpers ----------------------
function pad2($n){ return str_pad($n,2,'0',STR_PAD_LEFT); }

// Converte valor numérico para extenso (pt-BR) — cobre reais e centavos
function valor_por_extenso($valor) {
    // base simples e robusta
    $valor = number_format((float)$valor, 2, '.', '');
    list($reais, $centavos) = explode('.', $valor);

    $extenso = numero_por_extenso_int($reais) . ' real' . ($reais == '1' ? '' : 'es');
    if ((int)$centavos > 0) {
        $extenso .= ' e ' . numero_por_extenso_int($centavos) . ' centavo' . ((int)$centavos == 1 ? '' : 's');
    }
    return ucwords($extenso);
}

// converte inteiros (0..999999999...) para extenso simples (usa grupos de 3)
function numero_por_extenso_int($num) {
    $num = preg_replace('/[^0-9]/','',$num);
    if ($num === '' ) return 'zero';
    $n = (int)$num;
    if ($n === 0) return 'zero';

    $unidades = ["","um","dois","três","quatro","cinco","seis","sete","oito","nove","dez","onze","doze","treze","quatorze","quinze","dezesseis","dezessete","dezoito","dezenove"];
    $dezenas  = ["","","vinte","trinta","quarenta","cinquenta","sessenta","setenta","oitenta","noventa"];
    $centenas = ["","cem","duzentos","trezentos","quatrocentos","quinhentos","seiscentos","setecentos","oitocentos","novecentos"];
    $classes = ["","mil","milhão","bilhão","trilhão"];
    // split in groups of 3
    $num = str_pad($num, ceil(strlen($num)/3)*3, '0', STR_PAD_LEFT);
    $groups = str_split($num,3);
    $out = [];
    $gcount = count($groups);
    foreach ($groups as $i => $grp) {
        $val = (int)$grp;
        if ($val === 0) { continue; }
        $h = (int)($val/100);
        $t = $val % 100;
        $u = $val % 10;
        $part = '';
        if ($val == 100) $part = 'cem';
        else {
            if ($h) $part .= $centenas[$h];
            if ($t) {
                if ($part) $part .= ' e ';
                if ($t < 20) $part .= $unidades[$t];
                else {
                    $part .= $dezenas[(int)($t/10)];
                    if ($u) $part .= ' e ' . $unidades[$u];
                }
            }
        }
        $clsIndex = $gcount - $i - 1;
        $clsName = $classes[$clsIndex] ?? '';
        if ($clsName) {
            if ($val > 1 && $clsName == 'milhão') $clsName = 'milhões';
            if ($val > 1 && $clsName == 'bilhão') $clsName = 'bilhões';
            if ($val > 1 && $clsName == 'trilhão') $clsName = 'trilhões';
        }
        $out[] = trim($part . ($clsName ? ' ' . $clsName : ''));
    }
    // join with commas and ' e ' for last
    if (count($out) == 1) return $out[0];
    $last = array_pop($out);
    return implode(', ', $out) . ' e ' . $last;
}

// formata data pt-BR
function data_fmt($d) {
    if (!$d) return '';
    return date('d/m/Y', strtotime($d));
}

// obter nome do mês em PT
function nome_mes($m){
    $meses = ["","Janeiro","Fevereiro","Março","Abril","Maio","Junho","Julho","Agosto","Setembro","Outubro","Novembro","Dezembro"];
    return $meses[(int)$m] ?? '';
}

// ---------------------- parâmetros / defaults ----------------------
$professor_id = $_GET['professor'] ?? 'todos';
$data_inicio = $_GET['data_inicio'] ?? '';
$data_fim = $_GET['data_fim'] ?? '';

// se botão de mês enviado via GET 'mes' e 'ano', ajusta datas
if (!empty($_GET['mes']) && !empty($_GET['ano'])) {
    $mes = intval($_GET['mes']);
    $ano = intval($_GET['ano']);
    $data_inicio = date('Y-m-d', strtotime("$ano-" . pad2($mes) . "-01"));
    $data_fim = date('Y-m-d', strtotime("$ano-" . pad2($mes) . "-01 +1 month -1 day"));
}

// se não informado, padrão mês atual
if (empty($data_inicio) || empty($data_fim)) {
    $data_inicio = date('Y-m-01');
    $data_fim = date('Y-m-t');
}

// segurança básica
$data_inicio_db = $conn->real_escape_string($data_inicio);
$data_fim_db   = $conn->real_escape_string($data_fim);

// ---------------------- buscar professores a processar ----------------------
$professor_list = [];

if ($professor_id !== '' && $professor_id !== 'todos') {
    // único professor (se existir)
    $stmt = $conn->prepare("SELECT * FROM professores WHERE id = ?");
    $stmt->bind_param("i", $professor_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res && $res->num_rows) {
        $professor_list[] = $res->fetch_assoc();
    }
} else {
    // pegar todos que têm ponto no período (evita listar professores sem registros)
    $stmt = $conn->prepare("SELECT DISTINCT id_professor FROM ponto WHERE data_ponto BETWEEN ? AND ?");
    $stmt->bind_param("ss", $data_inicio_db, $data_fim_db);
    $stmt->execute();
    $res = $stmt->get_result();
    $ids = [];
    while ($r = $res->fetch_assoc()) $ids[] = (int)$r['id_professor'];
    if (!empty($ids)) {
        // buscar dados completos dos professores selecionados
        $in = implode(',', $ids);
        $q = "SELECT * FROM professores WHERE id IN ($in) ORDER BY nome";
        $res2 = $conn->query($q);
        while ($p = $res2->fetch_assoc()) $professor_list[] = $p;
    }
}

// ---------------------- cálculo por professor ----------------------
$dados = []; // array de resultados por prof
$total_seconds_geral = 0;
$total_valor_geral = 0.0;

$points_stmt = $conn->prepare("SELECT hora_entrada, hora_saida, data_ponto, observacoes FROM ponto WHERE id_professor = ? AND data_ponto BETWEEN ? AND ? ORDER BY data_ponto ASC");

foreach ($professor_list as $prof) {
    $prof_id = (int)$prof['id'];
    $points_stmt->bind_param("iss", $prof_id, $data_inicio_db, $data_fim_db);
    $points_stmt->execute();
    $res = $points_stmt->get_result();

    $total_seconds = 0;
    $detalhes = [];
    while ($row = $res->fetch_assoc()) {
        $entrada = $row['hora_entrada'];
        $saida = $row['hora_saida'];
        $seg = 0;
        if ($entrada && $saida) {
            // suporta formatos 'HH:MM' e 'HH:MM:SS'
            $ts_ent = strtotime($row['data_ponto'] . ' ' . $entrada);
            $ts_sai = strtotime($row['data_ponto'] . ' ' . $saida);
            // se hora de saída menor que entrada (possible cross midnight) — tentar ajustar por +1 dia
            if ($ts_sai <= $ts_ent) $ts_sai += 24*3600;
            $seg = max(0, $ts_sai - $ts_ent);
            $total_seconds += $seg;
        }
        $detalhes[] = [
            'data' => $row['data_ponto'],
            'entrada' => $entrada,
            'saida' => $saida,
            'segundos' => $seg,
            'observacoes' => $row['observacoes'] ?? ''
        ];
    }

    $horas = floor($total_seconds / 3600);
    $minutos = floor(($total_seconds % 3600) / 60);
    $decHoras = $total_seconds / 3600.0; // decimal exato
    $valor_hora = (float)$prof['valor_hora_aula'];
    $valor_total = round($decHoras * $valor_hora, 2);

    $dados[] = [
        'prof' => $prof,
        'detalhes' => $detalhes,
        'total_seconds' => $total_seconds,
        'horas' => $horas,
        'minutos' => $minutos,
        'decHoras' => $decHoras,
        'valor_hora' => $valor_hora,
        'valor_total' => $valor_total
    ];

    $total_seconds_geral += $total_seconds;
    $total_valor_geral += $valor_total;
}

// total geral em h/m
$total_horas_geral = floor($total_seconds_geral / 3600);
$total_minutos_geral = floor(($total_seconds_geral % 3600) / 60);

// ---------------------- design & output ----------------------
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Relatório de Pagamento de Instrutores</title>
<style>
:root{
  --bg:#f5f7fb;
  --card:#ffffff;
  --muted:#6b7280;
  --accent:#4f46e5;
  --success:#10b981;
  --danger:#ef4444;
  --border: #e6e9ee;
}
*{box-sizing:border-box}
body{font-family: "Inter", "Segoe UI", Roboto, Arial, sans-serif; background:var(--bg); margin:0; color:#0f172a}
.container{max-width:1100px; margin:28px auto; padding:20px;}
.header {display:flex; gap:16px; flex-wrap:wrap; align-items:center; margin-bottom:18px;}
.title {flex:1}
h1{margin:0; font-size:20px}
.controls {display:flex; gap:12px; flex-wrap:wrap; align-items:center}
.card-months {display:flex; gap:10px; flex-wrap:wrap; width:100%; margin-bottom:12px}
.month-card{background:var(--card); border:1px solid var(--border); padding:10px 14px; border-radius:10px; min-width:110px; text-align:center; box-shadow:0 6px 18px rgba(15,23,42,0.04); cursor:pointer; transition:transform .12s ease, box-shadow .12s ease}
.month-card:hover{transform:translateY(-4px); box-shadow:0 10px 30px rgba(15,23,42,0.08)}
.filter {display:flex; gap:10px; align-items:center; background:var(--card); padding:12px; border-radius:10px; border:1px solid var(--border)}
.select, .date {padding:8px 10px; border:1px solid var(--border); border-radius:8px; background:white}
.btn {padding:8px 12px; border-radius:8px; border:none; cursor:pointer}
.btn.primary{background:var(--accent); color:white}
.btn.ghost{background:transparent; border:1px solid var(--border)}
.summary {background:linear-gradient(180deg,#fff,#fbfdff); padding:16px; border-radius:12px; box-shadow:0 6px 18px rgba(15,23,42,0.03); margin:14px 0; border:1px solid var(--border)}
.summary-grid {display:flex; gap:12px; flex-wrap:wrap; align-items:center}
.summary-item{flex:1; min-width:180px; background:transparent}
.kv {font-size:14px; color:var(--muted); margin:0}
.v {font-size:18px; font-weight:600; margin:6px 0 0 0}

.prof-block {background:var(--card); padding:14px; border-radius:10px; margin-bottom:18px; border:1px solid var(--border)}
.prof-top {display:flex; justify-content:space-between; gap:8px; align-items:center}
.prof-name {font-weight:700}
.small {font-size:13px; color:var(--muted)}

.table {width:100%; border-collapse:collapse; margin-top:10px}
.table th, .table td {padding:8px 10px; border-bottom:1px solid #eef2f7; text-align:left; font-size:13px}
.receipt {background:linear-gradient(180deg,#fff,#fbfdff); padding:12px; border-radius:8px; margin-top:12px; border:1px dashed #d1d5db}
.cut {height:1px; border-top:2px dashed #cbd5e1; margin:24px 0}

/* print */
@media print {
  .no-print{display:none}
  body{background:white}
  .month-card{box-shadow:none}
  .cut{page-break-after:always}
}

/* responsiveness */
@media (max-width:720px){
  .summary-grid{flex-direction:column}
  .month-card{min-width:48%}
  .filter{flex-direction:column; align-items:stretch}
  .controls{flex-direction:column}
}
</style>
</head>
<body>
    <div class="app-container">
        <?php include 'menu.php'; // Nosso menu lateral ?>

        <main class="main-content">
            </main>

        <?php include 'footer.php'; // <<-- ADICIONE ESTA LINHA AQUI ?>
    </div>
<div class="container">
  <div class="header no-print">
    <div class="title">
      <br></br><h1>Relatório de Pagamento de Instrutores</h1>
      <div class="small">Período: <strong><?= data_fmt($data_inicio) ?></strong> a <strong><?= data_fmt($data_fim) ?></strong></div>
    </div>
    <div class="controls">
      <button class="btn ghost no-print" onclick="window.print()">Imprimir</button>
    </div>
  </div>

  <!-- meses em cards -->
  <div class="card-months no-print">
    <?php
    $ano_atual = date('Y');
    for ($m=1;$m<=12;$m++):
      $nome = nome_mes($m);
      // link para gerar mês
      $link = '?mes='.$m.'&ano='.$ano_atual.'&professor='.urlencode($professor_id);
    ?>
      <div class="month-card" onclick="location.href='<?= $link ?>'">
        <div style="font-weight:700"><?= $nome ?></div>
        <div class="small"><?= $ano_atual ?></div>
      </div>
    <?php endfor; ?>
  </div>

  <!-- filtro -->
  <form method="get" class="filter no-print" style="margin-bottom:12px">
    <select name="professor" class="select" aria-label="Professor">
      <option value="todos" <?= ($professor_id=='todos'?'selected':'') ?>>Todos</option>
      <?php
        $q = $conn->query("SELECT id, nome FROM professores ORDER BY nome");
        while ($r = $q->fetch_assoc()){
          $sel = ($professor_id == $r['id']) ? 'selected' : '';
          echo "<option value='".htmlspecialchars($r['id'])."' $sel>".htmlspecialchars($r['nome'])."</option>";
        }
      ?>
    </select>

    <input class="date" type="date" name="data_inicio" value="<?= htmlspecialchars($data_inicio) ?>">
    <input class="date" type="date" name="data_fim" value="<?= htmlspecialchars($data_fim) ?>">
    <button class="btn primary" type="submit">Gerar por Período</button>
  </form>

  <!-- resumo geral -->
  <div class="summary">
    <div class="summary-grid">
      <div class="summary-item">
        <div class="kv">Período</div>
        <div class="v"><?= data_fmt($data_inicio) ?> — <?= data_fmt($data_fim) ?></div>
      </div>
      <div class="summary-item">
        <div class="kv">Total de horas</div>
        <div class="v"><?= $total_horas_geral ?>h <?= $total_minutos_geral ?>min</div>
      </div>
      <div class="summary-item">
        <div class="kv">Total geral</div>
        <div class="v">R$ <?= number_format($total_valor_geral,2,',','.') ?></div>
      </div>
    </div>
  </div>

  <!-- por professor -->
  <?php if (empty($dados)): ?>
    <div class="prof-block">
      <div class="prof-top"><div class="prof-name">Nenhum registro de ponto encontrado neste período.</div></div>
    </div>
  <?php endif; ?>

  <?php foreach ($dados as $entry): 
     $p = $entry['prof'];
     $horas = $entry['horas'];
     $minutos = $entry['minutos'];
     $valor_total = $entry['valor_total'];
  ?>
    <div class="prof-block">
      <div class="prof-top">
        <div>
          <div class="prof-name"><?= htmlspecialchars($p['nome']) ?></div>
          <div class="small"><?= htmlspecialchars($p['escola']) ?> — CPF: <?= htmlspecialchars($p['cpf']) ?></div>
        </div>
        <div style="text-align:right">
          <div class="small">Valor hora/aula</div>
          <div style="font-weight:700">R$ <?= number_format($entry['valor_hora'],2,',','.') ?></div>
        </div>
      </div>

      <!-- relatório simples -->
      <div style="margin-top:12px">
        <div class="small" style="font-weight:700">Relatório de Simples Conferência</div>
        <div class="small">Período: <?= data_fmt($data_inicio) ?> a <?= data_fmt($data_fim) ?></div>
        <div style="margin-top:8px"><strong>Total de horas trabalhadas:</strong> <?= $horas ?>h <?= $minutos ?>min</div>
      </div>

      <!-- detalhes em tabela -->
      <?php if (!empty($entry['detalhes'])): ?>
        <table class="table" aria-label="Detalhes de ponto">
          <thead>
            <tr><th>Data</th><th>Entrada</th><th>Saída</th><th>Tempo (h:mm)</th><th>Observações</th></tr>
          </thead>
          <tbody>
            <?php foreach ($entry['detalhes'] as $d):
              $seg = $d['segundos'];
              $h = floor($seg/3600);
              $m = floor(($seg%3600)/60);
            ?>
              <tr>
                <td><?= data_fmt($d['data']) ?></td>
                <td><?= htmlspecialchars($d['entrada']) ?></td>
                <td><?= htmlspecialchars($d['saida']) ?></td>
                <td><?= $h ?>h <?= $m ?>m</td>
                <td><?= htmlspecialchars($d['observacoes']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>

      <!-- recibo -->
      <div class="receipt">
        <div style="font-size:15px;font-weight:700;margin-bottom:6px">Recibo</div>
        <div class="small">Eu, <strong><?= htmlspecialchars($p['nome']) ?></strong>, CPF <strong><?= htmlspecialchars($p['cpf']) ?></strong>, declaro ter recebido de <strong><?= htmlspecialchars($entry['prof']['escola']) ?></strong> o valor de <strong>R$ <?= number_format($valor_total,2,',','.') ?></strong> (<?= valor_por_extenso($valor_total) ?>), referente aos serviços prestados no período de <?= data_fmt($data_inicio) ?> a <?= data_fmt($data_fim) ?>.</div>
        <div style="margin-top:10px"><?= htmlspecialchars($entry['prof']['cidade']) ?>, <?= date('d') ?> de <?= nome_mes(date('n')) ?> de <?= date('Y') ?>.</div>
        <div style="margin-top:18px">__________________________________________<br>Assinatura</div>
      </div>

      <div class="cut"></div>
    </div>
  <?php endforeach; ?>

</div>
</body>
</html>
