<?php
// contrato_professor1.php (versão com fallbacks para encontrar a escola)
// Requisitos: fpdf.php e conexao.php no mesmo diretório

require __DIR__ . '/fpdf.php';
include __DIR__ . '/conexao.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Validação do id
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    die("ID inválido.");
}
$id = (int)$_GET['id'];

/* ---------- detectar possíveis campos de ligação em professores ---------- */
$allowed_school_fields = ['escola_id', 'id_escola', 'escola'];
$school_field = null;
$professor_columns = [];

$resCols = $conn->query("SHOW COLUMNS FROM professores");
if ($resCols) {
    while ($c = $resCols->fetch_assoc()) {
        $professor_columns[] = $c['Field'];
        if (in_array($c['Field'], $allowed_school_fields)) {
            $school_field = $c['Field'];
        }
    }
}

/* ---------- ler colunas da tabela 'escolas' ---------- */
$escolas_columns = [];
$res = $conn->query("SHOW COLUMNS FROM escolas");
if ($res) {
    while ($r = $res->fetch_assoc()) {
        $escolas_columns[] = $r['Field'];
    }
}

/* ---------- montar SELECT para pegar p.* e colunas de escolas que existam ---------- */
$select_fields_school = [];
$school_map = [
    'nome' => 'escola_nome',
    'cnpj' => 'escola_cnpj',
    'representante' => 'escola_representante',
    'rg_representante' => 'escola_rg_representante',
    'cpf_representante' => 'escola_cpf_representante',
    'endereco_completo' => 'escola_endereco_completo',
    'endereco' => 'escola_endereco',
    'cidade' => 'escola_cidade'
];

foreach ($school_map as $col => $alias) {
    if (in_array($col, $escolas_columns)) {
        $select_fields_school[] = "e.`$col` AS `$alias`";
    }
}

if ($school_field && count($select_fields_school) > 0) {
    $sql = "SELECT p.*, " . implode(", ", $select_fields_school) . "
            FROM professores p
            LEFT JOIN escolas e ON p.`{$school_field}` = e.id
            WHERE p.id = ?
            LIMIT 1";
} else {
    // sem join (ou sem colunas em escolas) pega só professor
    $sql = "SELECT p.* FROM professores p WHERE p.id = ? LIMIT 1";
}

/* ---------- preparar e executar ---------- */
$stmt = $conn->prepare($sql);
if (!$stmt) {
    die("Erro no prepare(): (" . $conn->errno . ") " . $conn->error . "\nSQL: " . $sql);
}
$stmt->bind_param("i", $id);
$stmt->execute();
$result = $stmt->get_result();
if (!$result) {
    die("Erro ao executar query: " . $conn->error);
}
if ($result->num_rows === 0) {
    die("Professor não encontrado.");
}
$prof = $result->fetch_assoc();

/* ---------- agora: se os dados da escola estiverem vazios, tentamos heurísticas ---------- */
$escolaEncontrada = false;
$escolaDados = [];

// Se a query já trouxe dados da escola (ex.: escola_nome), usamos
if (!empty($prof['escola_nome']) || !empty($prof['escola_cnpj']) || !empty($prof['escola_endereco_completo']) || !empty($prof['escola_cidade'])) {
    $escolaEncontrada = true;
    // já temos os alias no $prof (já serão usados abaixo)
} else {
    // 1) Tentar achar nome da escola em colunas do professor (texto)
    $possible_name_fields = ['escola', 'escola_nome', 'nome_escola', 'unidade', 'nome_unidade'];
    $searchName = null;
    foreach ($possible_name_fields as $f) {
        if (isset($prof[$f]) && trim($prof[$f]) !== '') {
            $searchName = trim($prof[$f]);
            break;
        }
    }

    if ($searchName) {
        // procurar por nome parecido na tabela escolas
        $q = "SELECT * FROM escolas WHERE nome LIKE ? LIMIT 1";
        $s = $conn->prepare($q);
        if ($s) {
            $like = "%{$searchName}%";
            $s->bind_param("s", $like);
            $s->execute();
            $r = $s->get_result();
            if ($r && $r->num_rows > 0) {
                $escolaDados = $r->fetch_assoc();
                $escolaEncontrada = true;
            }
            $s->close();
        }
    }
}

// 2) Se ainda não encontrou e se havia um campo de ligação com valor numérico, tentar buscar pelo id
if (!$escolaEncontrada && $school_field && isset($prof[$school_field]) && is_numeric($prof[$school_field]) && intval($prof[$school_field]) > 0) {
    $sid = (int)$prof[$school_field];
    $q2 = "SELECT * FROM escolas WHERE id = ? LIMIT 1";
    $s2 = $conn->prepare($q2);
    if ($s2) {
        $s2->bind_param("i", $sid);
        $s2->execute();
        $r2 = $s2->get_result();
        if ($r2 && $r2->num_rows > 0) {
            $escolaDados = $r2->fetch_assoc();
            $escolaEncontrada = true;
        }
        $s2->close();
    }
}

// 3) se ainda não achou, tentar se o professor tem campo cidade próprio
$cidade = '';
if (!empty($prof['escola_cidade'])) {
    $cidade = $prof['escola_cidade'];
} elseif ($escolaEncontrada && !empty($escolaDados['cidade'])) {
    $cidade = $escolaDados['cidade'];
} elseif (!empty($prof['cidade'])) {
    $cidade = $prof['cidade'];
} else {
    $cidade = ''; // deixamos em branco por enquanto; abaixo colocamos fallback 'Alvorada' se vazio
}

/* ---------- montar variáveis finais do contratante (com fallbacks) ---------- */
if ($escolaEncontrada && !empty($escolaDados)) {
    $contratante_nome = $escolaDados['nome'] ?? 'M&K CURSOS E TREINAMENTOS LTDA';
    $contratante_cnpj = $escolaDados['cnpj'] ?? '47.749.272/0001-52';
    $contratante_endereco = $escolaDados['endereco_completo'] ?? $escolaDados['endereco'] ?? '';
    $contratante_representante = $escolaDados['representante'] ?? '';
    $contratante_rg = $escolaDados['rg_representante'] ?? '';
    $contratante_cpf = $escolaDados['cpf_representante'] ?? '';
    $cidade = $escolaDados['cidade'] ?? $cidade;
} else {
    // tentar usar os alias que podem ter vindo via JOIN (quando não usamos $escolaDados)
    $contratante_nome = $prof['escola_nome'] ?? 'M&K CURSOS E TREINAMENTOS LTDA';
    $contratante_cnpj = $prof['escola_cnpj'] ?? '47.749.272/0001-52';
    $contratante_endereco = $prof['escola_endereco_completo'] ?? $prof['escola_endereco'] ?? '';
    $contratante_representante = $prof['escola_representante'] ?? '';
    $contratante_rg = $prof['escola_rg_representante'] ?? '';
    $contratante_cpf = $prof['escola_cpf_representante'] ?? '';
    $cidade = $prof['escola_cidade'] ?? $cidade;
}

// último fallback para cidade
$cidade = $cidade ?: ($prof['cidade'] ?? ($prof['prof_cidade'] ?? '') );
if (empty($cidade)) $cidade = 'Alvorada'; // você pode trocar esse fallback

/* ---------- função data por extenso (pt-BR) ---------- */
function data_por_extenso($date) {
    if (empty($date)) return '';
    $meses = ['janeiro','fevereiro','março','abril','maio','junho','julho','agosto','setembro','outubro','novembro','dezembro'];
    $ts = strtotime($date);
    if (!$ts) return '';
    $d = date('d', $ts);
    $m = (int)date('m', $ts) - 1;
    $y = date('Y', $ts);
    return sprintf("%02d de %s de %04d", $d, $meses[$m], $y);
}

/* ---------- datas e valores ---------- */
$data_inicio  = data_por_extenso($prof['data_admissao']);
$data_termino = data_por_extenso(date('Y-m-d', strtotime("+1 year", strtotime($prof['data_admissao']))));
$valor_hora_aula = "R$ " . number_format((float)($prof['valor_hora_aula'] ?? 0), 2, ',', '.');
$dia_pagamento = !empty($prof['dia_pagamento']) ? $prof['dia_pagamento'] : '10';


/* ---------- gerar PDF (FPDF) ---------- */
class PDF extends FPDF {
    function Header() {
        $logo = __DIR__ . '/logo.png';
        if (file_exists($logo)) {
            $this->Image($logo, 10, 6, 40);
        }
        $this->SetFont('Arial', 'B', 12);
        $this->Cell(0, 10, utf8_decode('CONTRATO PARTICULAR DE PRESTAÇÃO DE SERVIÇOS DE TRABALHO AUTÔNOMO'), 0, 1, 'C');
        $this->Ln(5);
    }
    function Footer() {
        $this->SetY(-15);
        $this->SetFont('Arial', 'I', 8);
        $this->Cell(0, 5, utf8_decode('Data de emissão: ' . date('d/m/Y')), 0, 1, 'L');
        $this->Cell(0, 5, utf8_decode('Página ') . $this->PageNo() . '/{nb}', 0, 0, 'R');
    }
    function ChapterBody($txt) {
        $this->SetFont('Arial', '', 11);
        $this->MultiCell(0, 6, utf8_decode($txt));
        $this->Ln();
    }
}

$pdf = new PDF();
$pdf->AliasNbPages();
$pdf->AddPage();

$cidade_assinatura = $cidade ?: 'Alvorada';

$texto = "
Pelo presente Instrumento Particular de Contrato de Prestação de Trabalho Autônomo, de um lado {$contratante_nome} devidamente inscrita no CNPJ nº {$contratante_cnpj}, situada na {$contratante_endereco}, neste ato devidamente representado por sua sócia proprietária, {$contratante_representante}, brasileira, casada, Administrador de Empresas, RG {$contratante_rg} e CPF {$contratante_cpf} doravante denominado CONTRATANTE, e de outro lado, {$prof['nome']}, {$prof['estado_civil']}, RG Nº {$prof['rg']} CPF Nº {$prof['cpf']} com endereço em {$prof['endereco']}, neste ato doravante denominado de CONTRATADO, têm entre si justo e contratado o presente contrato mediante as cláusulas e condições seguintes, que mutuamente prometem cumprir e respeitar, a saber:

CLÁUSULA PRIMEIRA: O presente contrato tem como objetivo a prestação de serviços de trabalho autônomo de por parte do CONTRATADO, e mais as funções que vierem a ser objeto de solicitações verbais ou por escrito, seguindo as necessidades da CONTRATANTE, desde que compatíveis com as atribuições da modalidade de especialização do CONTRATADO.

CLÁUSULA SEGUNDA: Os serviços ora contratados terão sua realização junto às dependências da CONTRATANTE, no supra citado endereço, podendo, contudo, a CONTRATANTE transferir o CONTRATADO para outra localidade onde possa exercer suas funções, quer a título temporário, quer a título definitivo, da qual o CONTRATADO expressa, neste ato, sua concordância.

CLÁUSULA TERCEIRA: O horário a ser cumprido pelo CONTRATADO para a realização de seus serviços poderá ser nos períodos da manhã, tarde ou noite, sempre condicionado ao preenchimento de controle de horas.

CLÁUSULA QUARTA: O presente Contrato terá sua validade por prazo de um ano, tendo seu início em {$data_inicio} e término em {$data_termino}, ficando facultado às partes a rescisão do presente contrato, sem qualquer multa, desde que previamente avisada a parte contrária com antecedência mínima de 30 dias, ocasião em que o CONTRATADO receberá pelos seus serviços prestados nos moldes do quanto convencionado na cláusula quinta do presente contrato. Em caso de descumprimento do período de 30 dias, deverá qualquer uma das partes efetuar o pagamento à outra do valor referente às horas que seriam ministradas durante esse período.

CLÁUSULA SEXTA: Em contraprestação dos serviços, o CONTRATANTE pagará ao CONTRATADO o valor de {$valor_hora_aula} por hora aula, pago no dia {$dia_pagamento} de cada mês subsequente à prestação do serviço e mediante apresentação de RPA ou Nota Fiscal por parte do CONTRATADO.

CLÁUSULA OITAVA: As partes elegem o foro da comarca de {$cidade_assinatura}, para dirimir eventuais dúvidas decorrentes do presente.

E, por estarem assim justos e contratados, assinam o presente instrumento em duas vias de igual teor e forma, na presença de duas testemunhas.

{$cidade_assinatura}, _____ de _________________________ de _____.

CONTRATANTE: ______________________________________
{$contratante_nome}

CONTRATADO: ______________________________________
{$prof['nome']}

Testemunhas:
1. _______________________________________________
2. _______________________________________________";

$pdf->ChapterBody($texto);

// nome de arquivo seguro
$safe_name = preg_replace('/[^A-Za-z0-9_\-]/', '_', $prof['nome']);
$filename = "Contrato_Professor_{$safe_name}.pdf";

$pdf->Output("I", $filename);
exit;
