import React, { useEffect, useState } from 'react';
import { 
  Monitor, 
  Users, 
  DollarSign, 
  Clock, 
  Building, 
  Smartphone,
  Instagram,
  Mail,
  MessageCircle,
  ChevronUp,
  Star,
  CheckCircle,
  ArrowRight
} from 'lucide-react';

function App() {
  const [scrolled, setScrolled] = useState(false);
  const [showBackToTop, setShowBackToTop] = useState(false);

  useEffect(() => {
    const handleScroll = () => {
      const scrollTop = window.scrollY;
      setScrolled(scrollTop > 50);
      setShowBackToTop(scrollTop > 300);
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const scrollToSection = (sectionId: string) => {
    const element = document.getElementById(sectionId);
    element?.scrollIntoView({ behavior: 'smooth' });
  };

  return (
    <div className="min-h-screen bg-gray-900 text-white overflow-x-hidden">
      {/* Header */}
      <header className={`fixed top-0 w-full z-50 transition-all duration-300 ${
        scrolled ? 'bg-gray-900/95 backdrop-blur-md shadow-lg' : 'bg-transparent'
      }`}>
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="text-2xl font-bold bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent">
              Prática Tecnologias
            </div>
            <nav className="hidden md:flex space-x-8">
              {['Home', 'Sobre', 'Serviços', 'Portfólio', 'Contato'].map((item) => (
                <button
                  key={item}
                  onClick={() => scrollToSection(item.toLowerCase().replace('ó', 'o').replace('í', 'i'))}
                  className="hover:text-cyan-400 transition-colors duration-300 font-medium"
                >
                  {item}
                </button>
              ))}
            </nav>
            <button className="md:hidden text-cyan-400">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
              </svg>
            </button>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section id="home" className="relative min-h-screen flex items-center justify-center overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-blue-900 via-purple-900 to-gray-900"></div>
          <div className="absolute inset-0 bg-[url('data:image/svg+xml,%3Csvg width="60" height="60" viewBox="0 0 60 60" xmlns="http://www.w3.org/2000/svg"%3E%3Cg fill="none" fill-rule="evenodd"%3E%3Cg fill="%23ffffff" fill-opacity="0.05"%3E%3Ccircle cx="7" cy="7" r="1"/%3E%3C/g%3E%3C/g%3E%3C/svg%3E')] opacity-20"></div>
        
        <div className="container mx-auto px-4 text-center relative z-10">
          <div className="animate-fade-in-up">
            <h1 className="text-5xl md:text-7xl font-bold mb-6 bg-gradient-to-r from-white via-cyan-400 to-blue-400 bg-clip-text text-transparent">
              Sistemas Práticos
            </h1>
            <p className="text-xl md:text-2xl text-gray-300 mb-8 max-w-3xl mx-auto leading-relaxed">
              Desenvolvemos soluções tecnológicas inteligentes para empresas que buscam praticidade, 
              eficiência e resultados excepcionais
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button 
                onClick={() => scrollToSection('contato')}
                className="bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 px-8 py-4 rounded-full text-lg font-semibold transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-cyan-500/25"
              >
                Solicitar Orçamento
              </button>
              <button 
                onClick={() => scrollToSection('sobre')}
                className="border-2 border-cyan-400 hover:bg-cyan-400 hover:text-gray-900 px-8 py-4 rounded-full text-lg font-semibold transition-all duration-300 transform hover:scale-105"
              >
                Conhecer Mais
              </button>
            </div>
          </div>
        </div>

        {/* Floating elements */}
        <div className="absolute top-1/4 left-10 animate-float">
          <div className="w-20 h-20 bg-gradient-to-br from-blue-500/20 to-cyan-500/20 rounded-full backdrop-blur-sm"></div>
        </div>
        <div className="absolute bottom-1/3 right-10 animate-float-delayed">
          <div className="w-16 h-16 bg-gradient-to-br from-purple-500/20 to-blue-500/20 rounded-full backdrop-blur-sm"></div>
        </div>
      </section>

      {/* About Section */}
      <section id="sobre" className="py-20 bg-gray-800">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-6 bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent">
              Quem Somos
            </h2>
            <div className="w-24 h-1 bg-gradient-to-r from-blue-600 to-cyan-600 mx-auto rounded-full"></div>
          </div>

          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div className="space-y-6">
              <p className="text-lg text-gray-300 leading-relaxed">
                A <span className="text-cyan-400 font-semibold">Prática Tecnologias</span> é especializada 
                em fornecer sistemas práticos e eficientes para empresas de todos os portes.
              </p>
              <p className="text-lg text-gray-300 leading-relaxed">
                Nossa missão é simplificar processos complexos através de soluções tecnológicas 
                inteligentes, oferecendo tanto sistemas prontos para uso quanto desenvolvimento 
                personalizado sob demanda.
              </p>
              <div className="grid grid-cols-2 gap-4">
                <div className="text-center p-4 bg-gradient-to-br from-blue-900/50 to-cyan-900/50 rounded-lg">
                  <div className="text-2xl font-bold text-cyan-400">100+</div>
                  <div className="text-sm text-gray-400">Projetos Entregues</div>
                </div>
                <div className="text-center p-4 bg-gradient-to-br from-purple-900/50 to-blue-900/50 rounded-lg">
                  <div className="text-2xl font-bold text-cyan-400">50+</div>
                  <div className="text-sm text-gray-400">Clientes Satisfeitos</div>
                </div>
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-4">
                <div className="bg-gradient-to-br from-blue-600/20 to-cyan-600/20 p-6 rounded-lg backdrop-blur-sm">
                  <CheckCircle className="w-8 h-8 text-cyan-400 mb-3" />
                  <h3 className="font-semibold mb-2">Sistemas Prontos</h3>
                  <p className="text-sm text-gray-400">Soluções testadas e aprovadas</p>
                </div>
                <div className="bg-gradient-to-br from-purple-600/20 to-blue-600/20 p-6 rounded-lg backdrop-blur-sm">
                  <Star className="w-8 h-8 text-cyan-400 mb-3" />
                  <h3 className="font-semibold mb-2">Qualidade Premium</h3>
                  <p className="text-sm text-gray-400">Excelência em cada detalhe</p>
                </div>
              </div>
              <div className="space-y-4 mt-8">
                <div className="bg-gradient-to-br from-cyan-600/20 to-blue-600/20 p-6 rounded-lg backdrop-blur-sm">
                  <Building className="w-8 h-8 text-cyan-400 mb-3" />
                  <h3 className="font-semibold mb-2">Para Empresas</h3>
                  <p className="text-sm text-gray-400">Focado em pessoas jurídicas</p>
                </div>
                <div className="bg-gradient-to-br from-blue-600/20 to-purple-600/20 p-6 rounded-lg backdrop-blur-sm">
                  <Monitor className="w-8 h-8 text-cyan-400 mb-3" />
                  <h3 className="font-semibold mb-2">Sob Demanda</h3>
                  <p className="text-sm text-gray-400">Desenvolvimento personalizado</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Services Section */}
      <section id="servicos" className="py-20 bg-gray-900">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-6 bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent">
              Nossos Serviços
            </h2>
            <div className="w-24 h-1 bg-gradient-to-r from-blue-600 to-cyan-600 mx-auto rounded-full mb-6"></div>
            <p className="text-xl text-gray-400 max-w-2xl mx-auto">
              Soluções completas para otimizar sua empresa
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                icon: Users,
                title: "Gestão de Professores e Ponto",
                description: "Sistema completo para controle de frequência e gestão de profissionais da educação"
              },
              {
                icon: Building,
                title: "Gestão de Empresa",
                description: "Plataforma integrada para administração completa do seu negócio"
              },
              {
                icon: DollarSign,
                title: "Sistemas Financeiros",
                description: "Controle financeiro avançado com relatórios detalhados e análises"
              },
              {
                icon: Clock,
                title: "Controle de Ponto",
                description: "Gestão automatizada de horários com relatórios precisos"
              },
              {
                icon: Monitor,
                title: "Sistemas Web",
                description: "Desenvolvimento de aplicações web modernas e responsivas"
              },
              {
                icon: Smartphone,
                title: "Apps Mobile",
                description: "Aplicativos nativos para Android e iOS sob medida"
              }
            ].map((service, index) => (
              <div 
                key={index}
                className="group bg-gradient-to-br from-gray-800 to-gray-700 p-8 rounded-xl hover:from-blue-900/50 hover:to-cyan-900/50 transition-all duration-500 transform hover:-translate-y-2 hover:shadow-2xl hover:shadow-cyan-500/20"
              >
                <service.icon className="w-12 h-12 text-cyan-400 mb-6 group-hover:scale-110 transition-transform duration-300" />
                <h3 className="text-xl font-bold mb-4 group-hover:text-cyan-400 transition-colors duration-300">
                  {service.title}
                </h3>
                <p className="text-gray-400 leading-relaxed group-hover:text-gray-300 transition-colors duration-300">
                  {service.description}
                </p>
                <div className="mt-6 flex items-center text-cyan-400 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                  <span className="text-sm font-medium">Saiba mais</span>
                  <ArrowRight className="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform duration-300" />
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Portfolio Section */}
      <section id="portfolio" className="py-20 bg-gray-800">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-6 bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent">
              Nosso Portfólio
            </h2>
            <div className="w-24 h-1 bg-gradient-to-r from-blue-600 to-cyan-600 mx-auto rounded-full mb-6"></div>
            <p className="text-xl text-gray-400 max-w-2xl mx-auto">
              Conheça alguns projetos que transformaram negócios
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                title: "Sistema de Gestão Escolar",
                description: "Plataforma completa para administração educacional com módulos integrados",
                image: "https://images.pexels.com/photos/256490/pexels-photo-256490.jpeg?auto=compress&cs=tinysrgb&w=500"
              },
              {
                title: "ERP Empresarial",
                description: "Sistema integrado de gestão para empresas de médio porte",
                image: "https://images.pexels.com/photos/590016/pexels-photo-590016.jpeg?auto=compress&cs=tinysrgb&w=500"
              },
              {
                title: "App de Controle Financeiro",
                description: "Aplicativo mobile para gestão financeira pessoal e empresarial",
                image: "https://images.pexels.com/photos/159888/pexels-photo-159888.jpeg?auto=compress&cs=tinysrgb&w=500"
              },
              {
                title: "Sistema de Ponto Eletrônico",
                description: "Controle de frequência com biometria e relatórios avançados",
                image: "https://images.pexels.com/photos/943096/pexels-photo-943096.jpeg?auto=compress&cs=tinysrgb&w=500"
              },
              {
                title: "Plataforma E-commerce",
                description: "Loja virtual completa com gestão integrada de estoque e vendas",
                image: "https://images.pexels.com/photos/230544/pexels-photo-230544.jpeg?auto=compress&cs=tinysrgb&w=500"
              },
              {
                title: "Sistema de CRM",
                description: "Gestão de relacionamento com clientes e automação de vendas",
                image: "https://images.pexels.com/photos/3184360/pexels-photo-3184360.jpeg?auto=compress&cs=tinysrgb&w=500"
              }
            ].map((project, index) => (
              <div 
                key={index}
                className="group relative overflow-hidden rounded-xl bg-gray-700 hover:transform hover:scale-105 transition-all duration-500 hover:shadow-2xl hover:shadow-cyan-500/20"
              >
                <div className="aspect-video overflow-hidden">
                  <img 
                    src={project.image} 
                    alt={project.title}
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
                  />
                </div>
                <div className="absolute inset-0 bg-gradient-to-t from-gray-900 via-gray-900/50 to-transparent opacity-80 group-hover:opacity-90 transition-opacity duration-300"></div>
                <div className="absolute bottom-0 left-0 right-0 p-6">
                  <h3 className="text-xl font-bold mb-2 text-white group-hover:text-cyan-400 transition-colors duration-300">
                    {project.title}
                  </h3>
                  <p className="text-gray-300 text-sm leading-relaxed">
                    {project.description}
                  </p>
                  <div className="mt-4 flex items-center text-cyan-400 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                    <span className="text-sm font-medium">Ver projeto</span>
                    <ArrowRight className="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform duration-300" />
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Instagram CTA Section */}
      <section className="py-16 bg-gradient-to-r from-purple-900 via-blue-900 to-cyan-900">
        <div className="container mx-auto px-4 text-center">
          <Instagram className="w-16 h-16 text-white mx-auto mb-6" />
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Siga-nos no Instagram
          </h2>
          <p className="text-xl text-gray-200 mb-8 max-w-2xl mx-auto">
            Acompanhe nossos projetos, dicas de tecnologia e novidades do mundo digital
          </p>
          <a
            href="https://instagram.com/praticatecnologias"
            target="_blank"
            rel="noopener noreferrer"
            className="inline-flex items-center gap-3 bg-gradient-to-r from-pink-600 to-purple-600 hover:from-pink-700 hover:to-purple-700 px-8 py-4 rounded-full text-lg font-semibold transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-purple-500/25"
          >
            <Instagram className="w-6 h-6" />
            @praticatecnologias
          </a>
        </div>
      </section>

      {/* Contact Section */}
      <section id="contato" className="py-20 bg-gray-900">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-6 bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent">
              Entre em Contato
            </h2>
            <div className="w-24 h-1 bg-gradient-to-r from-blue-600 to-cyan-600 mx-auto rounded-full mb-6"></div>
            <p className="text-xl text-gray-400 max-w-2xl mx-auto">
              Vamos transformar sua ideia em realidade
            </p>
          </div>

          <div className="grid md:grid-cols-2 gap-12 max-w-6xl mx-auto">
            <div className="space-y-8">
              <h3 className="text-2xl font-bold mb-6">Fale Conosco</h3>
              
              <div className="space-y-6">
                <div className="flex items-center gap-4 p-4 bg-gradient-to-r from-blue-900/30 to-cyan-900/30 rounded-lg">
                  <MessageCircle className="w-8 h-8 text-cyan-400" />
                  <div>
                    <h4 className="font-semibold">WhatsApp</h4>
                    <p className="text-gray-400">+55 51 99766-1588</p>
                  </div>
                </div>
                
                <div className="flex items-center gap-4 p-4 bg-gradient-to-r from-purple-900/30 to-blue-900/30 rounded-lg">
                  <Mail className="w-8 h-8 text-cyan-400" />
                  <div>
                    <h4 className="font-semibold">E-mail</h4>
                    <p className="text-gray-400">contato@praticatecnologias.com.br</p>
                  </div>
                </div>
                
                <div className="flex items-center gap-4 p-4 bg-gradient-to-r from-cyan-900/30 to-blue-900/30 rounded-lg">
                  <Instagram className="w-8 h-8 text-cyan-400" />
                  <div>
                    <h4 className="font-semibold">Instagram</h4>
                    <p className="text-gray-400">@praticatecnologias</p>
                  </div>
                </div>
              </div>
            </div>

            <form className="space-y-6">
              <div>
                <label htmlFor="name" className="block text-sm font-medium mb-2">Nome da Empresa</label>
                <input
                  type="text"
                  id="name"
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg focus:ring-2 focus:ring-cyan-500 focus:border-transparent transition-colors duration-300"
                  placeholder="Digite o nome da sua empresa"
                />
              </div>
              
              <div>
                <label htmlFor="email" className="block text-sm font-medium mb-2">E-mail Corporativo</label>
                <input
                  type="email"
                  id="email"
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg focus:ring-2 focus:ring-cyan-500 focus:border-transparent transition-colors duration-300"
                  placeholder="empresa@exemplo.com.br"
                />
              </div>
              
              <div>
                <label htmlFor="phone" className="block text-sm font-medium mb-2">Telefone/WhatsApp</label>
                <input
                  type="tel"
                  id="phone"
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg focus:ring-2 focus:ring-cyan-500 focus:border-transparent transition-colors duration-300"
                  placeholder="(51) 99999-9999"
                />
              </div>
              
              <div>
                <label htmlFor="message" className="block text-sm font-medium mb-2">Mensagem</label>
                <textarea
                  id="message"
                  rows={5}
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg focus:ring-2 focus:ring-cyan-500 focus:border-transparent transition-colors duration-300 resize-none"
                  placeholder="Descreva seu projeto ou necessidade..."
                ></textarea>
              </div>
              
              <button
                type="submit"
                className="w-full bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 px-8 py-4 rounded-lg font-semibold transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-cyan-500/25"
              >
                Enviar Mensagem
              </button>
            </form>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-800 py-12 border-t border-gray-700">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-4 gap-8 mb-8">
            <div>
              <h3 className="text-xl font-bold bg-gradient-to-r from-blue-400 to-cyan-400 bg-clip-text text-transparent mb-4">
                Prática Tecnologias
              </h3>
              <p className="text-gray-400 text-sm leading-relaxed">
                Especialistas em sistemas práticos para empresas. 
                Transformamos ideias em soluções digitais eficientes.
              </p>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Serviços</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li>Gestão Empresarial</li>
                <li>Sistemas Financeiros</li>
                <li>Controle de Ponto</li>
                <li>Desenvolvimento Web</li>
              </ul>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Links Rápidos</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li><button onClick={() => scrollToSection('sobre')}>Sobre</button></li>
                <li><button onClick={() => scrollToSection('servicos')}>Serviços</button></li>
                <li><button onClick={() => scrollToSection('portfolio')}>Portfólio</button></li>
                <li><button onClick={() => scrollToSection('contato')}>Contato</button></li>
              </ul>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Contato</h4>
              <div className="space-y-2 text-sm text-gray-400">
                <p>WhatsApp: +55 51 99766-1588</p>
                <p>E-mail: contato@praticatecnologias.com.br</p>
                <p>Instagram: @praticatecnologias</p>
              </div>
            </div>
          </div>
          
          <div className="border-t border-gray-700 pt-8 text-center text-sm text-gray-400">
            <p>&copy; 2025 Prática Tecnologias. Todos os direitos reservados.</p>
            <p className="mt-2">Desenvolvido com ❤️ para transformar negócios através da tecnologia.</p>
          </div>
        </div>
      </footer>

      {/* WhatsApp Float Button */}
      <a
        href="https://wa.me/5551997661588?text=Olá! Gostaria de saber mais sobre os sistemas da Prática Tecnologias."
        target="_blank"
        rel="noopener noreferrer"
        className="fixed bottom-6 right-6 bg-green-500 hover:bg-green-600 text-white p-4 rounded-full shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-110 z-50 animate-pulse"
        title="WhatsApp"
      >
        <MessageCircle className="w-6 h-6" />
      </a>

      {/* Back to Top Button */}
      {showBackToTop && (
        <button
          onClick={scrollToTop}
          className="fixed bottom-6 left-6 bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 text-white p-3 rounded-full shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-110 z-50"
          title="Voltar ao topo"
        >
          <ChevronUp className="w-5 h-5" />
        </button>
      )}
    </div>
  );
}

export default App;