<?php
// Contact Form Handler for Prática Tecnologias
// This file handles the contact form submission

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die('Method not allowed');
}

// Configuration
$config = [
    'to_email' => 'contato@praticatecnologias.com.br',
    'from_email' => 'noreply@praticatecnologias.com.br',
    'subject_prefix' => '[Site] Novo contato - ',
    'success_redirect' => 'index.html#contato',
    'error_redirect' => 'index.html#contato'
];

// Input validation and sanitization
function validateAndSanitize($data) {
    $errors = [];
    $clean_data = [];
    
    // Name validation
    if (empty($data['name']) || strlen(trim($data['name'])) < 2) {
        $errors[] = 'Nome da empresa é obrigatório (mínimo 2 caracteres)';
    } else {
        $clean_data['name'] = htmlspecialchars(trim($data['name']), ENT_QUOTES, 'UTF-8');
    }
    
    // Email validation
    if (empty($data['email']) || !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'E-mail válido é obrigatório';
    } else {
        $clean_data['email'] = filter_var(trim($data['email']), FILTER_SANITIZE_EMAIL);
    }
    
    // Phone validation
    if (empty($data['phone']) || strlen(preg_replace('/[^0-9]/', '', $data['phone'])) < 10) {
        $errors[] = 'Telefone válido é obrigatório (mínimo 10 dígitos)';
    } else {
        $clean_data['phone'] = htmlspecialchars(trim($data['phone']), ENT_QUOTES, 'UTF-8');
    }
    
    // Message validation
    if (empty($data['message']) || strlen(trim($data['message'])) < 10) {
        $errors[] = 'Mensagem é obrigatória (mínimo 10 caracteres)';
    } else {
        $clean_data['message'] = htmlspecialchars(trim($data['message']), ENT_QUOTES, 'UTF-8');
    }
    
    return ['errors' => $errors, 'data' => $clean_data];
}

// Anti-spam protection
function isSpam($data) {
    // Check for common spam patterns
    $spam_patterns = [
        '/\b(viagra|cialis|casino|poker|loan|credit)\b/i',
        '/\b(click here|visit now|act now)\b/i',
        '/http[s]?:\/\/[^\s]+/i' // URLs in message
    ];
    
    $text_to_check = $data['name'] . ' ' . $data['message'];
    
    foreach ($spam_patterns as $pattern) {
        if (preg_match($pattern, $text_to_check)) {
            return true;
        }
    }
    
    // Check for excessive repetition
    if (substr_count(strtolower($text_to_check), strtolower($data['name'])) > 3) {
        return true;
    }
    
    return false;
}

// Rate limiting (simple file-based)
function checkRateLimit($ip) {
    $rate_limit_file = sys_get_temp_dir() . '/contact_rate_limit.json';
    $current_time = time();
    $rate_limit = 5; // Max 5 submissions per hour
    $time_window = 3600; // 1 hour
    
    // Load existing data
    $rate_data = [];
    if (file_exists($rate_limit_file)) {
        $rate_data = json_decode(file_get_contents($rate_limit_file), true) ?: [];
    }
    
    // Clean old entries
    $rate_data = array_filter($rate_data, function($timestamp) use ($current_time, $time_window) {
        return ($current_time - $timestamp) < $time_window;
    });
    
    // Check current IP
    $ip_submissions = array_filter($rate_data, function($timestamp, $stored_ip) use ($ip) {
        return $stored_ip === $ip;
    }, ARRAY_FILTER_USE_BOTH);
    
    if (count($ip_submissions) >= $rate_limit) {
        return false;
    }
    
    // Add current submission
    $rate_data[$ip] = $current_time;
    
    // Save data
    file_put_contents($rate_limit_file, json_encode($rate_data));
    
    return true;
}

// Email sending function
function sendEmail($data, $config) {
    $to = $config['to_email'];
    $subject = $config['subject_prefix'] . $data['name'];
    
    // Email body
    $body = "
    <html>
    <head>
        <title>Novo contato do site</title>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #0066cc, #00ffff); color: white; padding: 20px; text-align: center; }
            .content { background: #f9f9f9; padding: 20px; }
            .field { margin-bottom: 15px; }
            .label { font-weight: bold; color: #0066cc; }
            .value { margin-top: 5px; }
            .footer { background: #333; color: white; padding: 15px; text-align: center; font-size: 12px; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>Novo Contato - Prática Tecnologias</h2>
            </div>
            <div class='content'>
                <div class='field'>
                    <div class='label'>Empresa:</div>
                    <div class='value'>{$data['name']}</div>
                </div>
                <div class='field'>
                    <div class='label'>E-mail:</div>
                    <div class='value'>{$data['email']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Telefone:</div>
                    <div class='value'>{$data['phone']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Mensagem:</div>
                    <div class='value'>" . nl2br($data['message']) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Data/Hora:</div>
                    <div class='value'>" . date('d/m/Y H:i:s') . "</div>
                </div>
                <div class='field'>
                    <div class='label'>IP:</div>
                    <div class='value'>" . $_SERVER['REMOTE_ADDR'] . "</div>
                </div>
            </div>
            <div class='footer'>
                <p>Este e-mail foi enviado automaticamente pelo site praticatecnologias.com.br</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    // Headers
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: ' . $config['from_email'],
        'Reply-To: ' . $data['email'],
        'X-Mailer: PHP/' . phpversion(),
        'X-Priority: 3',
        'X-MSMail-Priority: Normal'
    ];
    
    return mail($to, $subject, $body, implode("\r\n", $headers));
}

// Log function for debugging
function logMessage($message, $level = 'INFO') {
    $log_file = sys_get_temp_dir() . '/contact_form.log';
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'];
    $log_entry = "[{$timestamp}] [{$level}] [{$ip}] {$message}" . PHP_EOL;
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
}

// Main processing
try {
    // Get client IP
    $client_ip = $_SERVER['REMOTE_ADDR'];
    
    // Check rate limiting
    if (!checkRateLimit($client_ip)) {
        logMessage("Rate limit exceeded for IP: {$client_ip}", 'WARNING');
        http_response_code(429);
        header('Location: ' . $config['error_redirect'] . '?error=rate_limit');
        exit;
    }
    
    // Validate and sanitize input
    $validation_result = validateAndSanitize($_POST);
    
    if (!empty($validation_result['errors'])) {
        logMessage("Validation errors: " . implode(', ', $validation_result['errors']), 'ERROR');
        $error_msg = urlencode(implode('. ', $validation_result['errors']));
        header('Location: ' . $config['error_redirect'] . '?error=' . $error_msg);
        exit;
    }
    
    $clean_data = $validation_result['data'];
    
    // Anti-spam check
    if (isSpam($clean_data)) {
        logMessage("Spam detected from IP: {$client_ip}", 'WARNING');
        // Don't reveal spam detection to user
        header('Location: ' . $config['success_redirect'] . '?success=1');
        exit;
    }
    
    // Send email
    if (sendEmail($clean_data, $config)) {
        logMessage("Email sent successfully for: {$clean_data['email']}", 'INFO');
        
        // Optional: Save to database
        // saveToDatabase($clean_data);
        
        header('Location: ' . $config['success_redirect'] . '?success=1');
    } else {
        logMessage("Failed to send email for: {$clean_data['email']}", 'ERROR');
        header('Location: ' . $config['error_redirect'] . '?error=email_failed');
    }
    
} catch (Exception $e) {
    logMessage("Exception: " . $e->getMessage(), 'ERROR');
    http_response_code(500);
    header('Location: ' . $config['error_redirect'] . '?error=server_error');
}

// Optional: Database storage function
function saveToDatabase($data) {
    try {
        // Database configuration
        $db_config = [
            'host' => 'localhost',
            'dbname' => 'pratica_tecnologias',
            'username' => 'your_username',
            'password' => 'your_password'
        ];
        
        $pdo = new PDO(
            "mysql:host={$db_config['host']};dbname={$db_config['dbname']};charset=utf8mb4",
            $db_config['username'],
            $db_config['password'],
            [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ]
        );
        
        $stmt = $pdo->prepare("
            INSERT INTO contacts (name, email, phone, message, ip_address, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $data['name'],
            $data['email'],
            $data['phone'],
            $data['message'],
            $_SERVER['REMOTE_ADDR']
        ]);
        
        return true;
        
    } catch (PDOException $e) {
        logMessage("Database error: " . $e->getMessage(), 'ERROR');
        return false;
    }
}

/*
-- SQL for creating contacts table (optional)
CREATE TABLE contacts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(50) NOT NULL,
    message TEXT NOT NULL,
    ip_address VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_created_at (created_at)
);
*/
?>